# 				lastfm2lastFM.pl  v0.3(MAY-2014)
#
#    Copyright (c) 2014 Lionel Grenier (lastfm2itunes@igrenier.com / lastfm.igrenier.com)
#
#    code for processing lastFM data has been inspired by Jugdizh (lastFM user)
#    .scrobbler.log format can be found here http://www.audioscrobbler.net/wiki/Portable_Player_Logging
#    .scrobbler.log submitting tools can be found here http://www.rockbox.org/twiki/bin/view/Main/LastFMLog
#  
#
#    WARNING: 
#            I STRONGLY RECOMMEND TO BACKUP YOUR iTUNES LIBRARY 
#            PRIOR TO USING THIS SCRIPT (just in case)
#    
# 
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# needed perl packages
use Class::Struct;
use XML::DOM;
use strict;
use POSIX qw(strftime);
use Encode;
use Time::Local;

binmode(STDOUT, ":utf8");

# script version
my $scriptVersion='v0.3 (MAY-2014)';

# lastFM URL
my $chartlisturl = 'http://ws.audioscrobbler.com/2.0/user/<username>/weeklychartlist.xml';
my $weeklycharturl = 'http://ws.audioscrobbler.com/2.0/user/<username>/weeklytrackchart.xml?from=<start>&to=<end>';

# lastFM username source
my $usernameS;
my $parser = new XML::DOM::Parser;

# lastFM username targetr
my $usernameT;

# lastFM data retrieval
my %trackplays;
my %trackdate;

# stats data
my $overAllCount = 0;
my $trackSkip=0;
my $trackNotFound=0;
my $overalltrack=0;
my $startDate='';
my $overallArtistsCount=0;
my $overallSongsCount=0;
my $overAllTrackCount=0;

# 1: debug messages to ON
# 0: debug messages to OFF
my $verbose = '';

# os supported
my $os='win';

my $file='';
my $outfilename='';
my $outfilenameLOG='';
my $time=0;

# lastFM track data structure
struct lastFMTrackInfo => {
	# Artist's name for current song
	songArtist => '$',
	# Track title for current song
	songTrack => '$',
	# Album title for current song.
	songAlbum => '$',
	# last played date
	lastPlayed => '$',
    # first played date
	firstPlayed => '$',
	# play count
	playedCount => '$',
	#  play time
	interpolatedPlayedTime => '$',
};

#############################################################"
# MAIN
print STDOUT "\nlastFM to lastFM track updater [".$scriptVersion."]\n";
print STDOUT "Copyright (c) 2014 Lionel Grenier (lastfm2itunes\@igrenier.com - lastfm.igrenier.com)\n";
print STDOUT "(if you have any question or problem do not hesitate to contact the author)\n";

# retrieve lastFM username
$usernameS = $ARGV[0] || die "\nERROR : Please supply your source last.fm username as the first argument.";
$outfilenameLOG="scrobbler.log";

my $dd = $ARGV[1] || '.';
if ($dd  eq "t") {
	$dd  = " ".$ARGV[2]."/".$ARGV[3]."/".$ARGV[4];
	print STDOUT "Start parsing for week around ".$dd."\n"; 
	$time = timelocal(0,0,0,$ARGV[2],$ARGV[3],$ARGV[4]);
	print STDOUT "Start parsing for week around ".strftime ("%d-%m-%Y", localtime($time));
	$verbose = $ARGV[5] || '.';
	if ($verbose eq "v") {
		print STDOUT "\tverbose on\n";
	} else {
		if ($verbose eq "csv") {
			$file="csv";
			$outfilename = "".$usernameS.".csv";
			$verbose="v";
		}
	}
} else {
	$verbose = $ARGV[1] || '.';
	if ($verbose eq "v") {
		print STDOUT "\tverbose on\n";
	} else {
		if ($verbose eq "csv") {
			$file="csv";
			$outfilename = "".$usernameS.".csv";
			$verbose="v";
		}
	}
}


print STDOUT "\n\n";

processLastFMTracks();

if ($file eq "csv") {
	writeToCSV();
} 

displayCSVStats();

print STDOUT "\nThanks for using the script. Bye\n";

exit;
#end of main
#############################################################"

#############################################################"
# lastFM sub functions  

sub processLastFMTracks {
	my $url = $chartlisturl;
	$url =~ s/<username>/$usernameS/g;
	print STDOUT "\nURL: ".$url."\n";
	
	my $chartlistdoc = $parser->parsefile($url);
	die "Could not retrieve page" unless $chartlistdoc;
	sleep 1;

	die "Could not find root node <weeklychartlist>" unless (my $rootlist = $chartlistdoc->getElementsByTagName("weeklychartlist",0))->getLength;
	my $chartlist = $rootlist->item(0)->getElementsByTagName("chart",0);
	
	print STDOUT "\n===================================\n";
	print STDOUT "\Creating lastFM log file: ".$outfilenameLOG." (lastFM format).\n";
	open OUTFILE, ">$outfilenameLOG" or die "Could not open file to write";
    # write file header 
	print OUTFILE "#AUDIOSCROBBLER/1.1\n";
	print OUTFILE "#TZ/UTC\n";
	print OUTFILE "#CLIENT/lastFM2lastFM 0.3\n";
	
	print STDOUT "\n=============================\n";
	print STDOUT "\tParsing lastFM data\n";
	print STDOUT "\tUsing ".$usernameS." lastFM account.\n";
	print STDOUT "\tFound a total of ".$chartlist->getLength." weeks full of played tracks to parse.\n\n";
	
	print STDOUT "\tLook for start date data\n";
	my $idx = -1;
	my $maxEnd = 0;
	
	for (my $i = 0; $i < $chartlist->getLength; $i++) {
	  my $start = $chartlist->item($i)->getAttribute("from") || die "ERROR: Could not find 'from' attribute in <chart> node.";
	  my $end = $chartlist->item($i)->getAttribute("to") || die "ERROR: Could not find 'to' attribute in <chart> node.";
	  $maxEnd = $end;
	  if ( $time>=$start && $time<=$end && $idx==-1) {
		$idx=$i;
		OutMsg("Found Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...\n");
	  }
	  
	  if ( $time<=$start && $idx==-1) {
		$idx=$i;
		OutMsg("Found Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...\n");
	  }
	}
	return print STDOUT "No data after the ".strftime ("%d-%m-%Y", localtime($time))." max available date ".strftime ("%d-%m-%Y", localtime($maxEnd)) unless $idx>-1;
	
	print STDOUT "\tWait while processing data\n";
	my $nbremain;
	
	for (my $i = $idx; $i < $chartlist->getLength; $i++) {
	  my $start = $chartlist->item($i)->getAttribute("from") || die "ERROR: Could not find 'from' attribute in <chart> node.";
	  my $end = $chartlist->item($i)->getAttribute("to") || die "ERROR: Could not find 'to' attribute in <chart> node.";
	 
	  OutMsg("Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...");
	  
	  $url = $weeklycharturl;
	  $url =~ s/<username>/$usernameS/g;
	  $url =~ s/<start>/$start/g;
	  $url =~ s/<end>/$end/g;
	  
	  print STDOUT "\nWeekly URL: ".$url."\n";
	  my $weeklychartdoc;
	  eval { $weeklychartdoc = $parser->parsefile($url); };
	  if (!$weeklychartdoc) {
		return;
	  }


	  return print STDOUT "Could not find root tag <weeklytrackchart>" unless ($rootlist = $weeklychartdoc->getElementsByTagName("weeklytrackchart",0))->getLength;
	  my $tracklist = $rootlist->item(0)->getElementsByTagName("track",0);
	  my $nbTracks = $tracklist->getLength;
	  
	  OutMsg("got ".$nbTracks." tracks this week\n");
	  
	  my $totalPlayCount = 0;
	  my $totalplays = 0;
	  my $trackCounter = 0;
	  # retrieve number of plays for that week
	  for (my $j = 0; $j < $tracklist->getLength; $j++) {
		return print STDOUT "Could not find <artist> tag" unless (my $artistlist = $tracklist->item($j)->getElementsByTagName("artist",0))->getLength;
	    return print STDOUT "Could not find <name> tag" unless (my $songlist = $tracklist->item($j)->getElementsByTagName("name",0))->getLength;
	    return print STDOUT "Could not find <playcount> tag" unless (my $playcountlist = $tracklist->item($j)->getElementsByTagName("playcount",0))->getLength;
		
		return print STDOUT "Could not get text from <artist> tag"
	      unless $artistlist->item(0)->hasChildNodes && (my $artisttext = $artistlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;
	    return print STDOUT "Could not get text from <name> tag"
	      unless $songlist->item(0)->hasChildNodes && (my $songtext = $songlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;
	    return print STDOUT "Could not get text from <playcount> tag"
	      unless $playcountlist->item(0)->hasChildNodes && (my $playcounttext = $playcountlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;
		
		my $artist = encode_utf8(lc($artisttext->getData));
		my $song = encode_utf8(lc($songtext->getData)); 
	
		
	    my $playcount = $playcounttext->getData;
		# keep the first date in lastFM data
		if ($startDate=='' && $playcount>0) {
			$startDate=strftime ("%d-%m-%Y", localtime($start));
			OutMsg(">>>>FOUND when you started scrobbling !!! on".$startDate."\n");
		}
		
		OutMsg("\t".$song." by ".$artist." played " .$playcount." times that week\n");
		
		if (exists $trackplays{$artist}) {
            if (exists $trackplays{$artist}{$song}) {
                $trackplays{$artist}{$song} += $playcount;
                $trackdate{$artist}{$song}{'END'} = strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
            } else {
                $trackplays{$artist}{$song} = $playcount;
                $trackdate{$artist}{$song}{'START'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($start));
                $trackdate{$artist}{$song}{'END'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
            }
	    } else {
	      $trackplays{$artist}{$song} = $playcount;
          $trackdate{$artist}{$song}{'START'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($start));
          $trackdate{$artist}{$song}{'END'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
	    }
	    $totalplays += $playcount;
		if ($totalPlayCount > 0) {
			my $elapsedTime = int(($end-$start)/$playcount);
			for (my $k = 0; $k < $playcount; $k++) {
				$trackCounter++;
				my $playTime=$start+$elapsedTime*$trackCounter;
				$trackdate{$artist}{$song}{'PLAYEDTIME'}{$k}=strftime ("%Y-%m-%d %H:%M:%S", localtime($playTime));
				
				print OUTFILE $artist;
				print OUTFILE "\t\t";
				print OUTFILE $song;
				print OUTFILE "\t\t";
				print OUTFILE $elapsedTime-1;
				print OUTFILE "\t";
				print OUTFILE "L";
				print OUTFILE "\t";
				print OUTFILE $playTime;
				print OUTFILE "\t\n";	

				#OutMsg("Interpolated Play Time: ",strftime ("%Y-%m-%d %H:%M:%S", localtime($playTime)),"\n");	
			}
		}
	  }
	  
	   
	  $overAllCount += $totalplays;
	  $overAllTrackCount = $overAllTrackCount+$tracklist->getLength;
	  $nbremain = $chartlist->getLength-($i+1);
	  OutMsg("got ".$totalplays." scrobbles this week for ".$tracklist->getLength." tracks. Overall ".$overAllCount." scrobbles for ".$overAllTrackCount." tracks so far..." .$nbremain." weeks remain to process\n");
	 
	}
	print STDOUT "\tParsing done\n";
	print STDOUT "=============================\n";
	
	close OUTFILE;
	print STDOUT "\nData written into .scrobbler.LOG\n";
	print STDOUT "\n===================================\n";
}

# handle output message
sub OutMsg {
	# Parameter - Message to be displayed
	my $message = join '',@_;

	print STDOUT $message;
	#if ($verbose eq "v")
	#{
	#	print STDOUT $message;      
	#}
}

# handle output error message
sub OutErrMsg {
	my $message = join '','Error: ',@_;
	print STDOUT $message; 
}

# display some stats about the script
sub displayStats {
	print STDOUT "\n=============================\n";
	print STDOUT "Statistics of lastFM 2 Itunes run for ".$usernameS." user account\n";
	print STDOUT "- ".$overAllTrackCount." tracks processed\n";
	print STDOUT "- ".$trackSkip." tracks skipped iTunes library\n";
	print STDOUT "- ".$trackNotFound." tracks not found in local iTunes library\n";
	print STDOUT "- you have ".$overAllCount." play count on lastFM since ".$startDate."\n";
	print STDOUT "=============================\n";
}
	
sub displayCSVStats {
	print STDOUT "\n=============================\n";
	print STDOUT "Statistics of lastFM for ".$usernameS." user account\n";
	print STDOUT "- ".$overAllTrackCount." tracks processed\n";
	print STDOUT "- ".$overallArtistsCount." artists processed\n";
	print STDOUT "- ".$overallSongsCount." songs processed\n";
	print STDOUT "- you have ".$overAllCount." play count on lastFM since ".$startDate."\n";
	print STDOUT "=============================\n";
}

sub writeToCSV {
	print STDOUT "\n===================================\n";
	print STDOUT "\nWrite tracks info into ".$outfilename." ($file format).\n";
	
	open OUTFILE, ">$outfilename" or die "Could not open file to write";
	my @artists = sort(keys %trackplays);
	print OUTFILE "Artist\tSong\tPlayCount\tFirstTime\tLastTime\n";
	for my $artist (@artists) {
		my @songs = sort(keys %{$trackplays{$artist}});
		for my $song (@songs) {
			my $lastFMtrack = lastFMTrackInfo->new();
			$lastFMtrack->songArtist($artist);
			$lastFMtrack->songTrack($song);
            $lastFMtrack->firstPlayed($trackdate{$artist}{$song}{'START'});
			$lastFMtrack->lastPlayed($trackdate{$artist}{$song}{'END'});
			$lastFMtrack->playedCount($trackplays{$artist}{$song});
			
			$lastFMtrack->interpolatedPlayedTime($time);
			print OUTFILE $lastFMtrack->songArtist;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->songTrack;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->playedCount;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->firstPlayed;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->lastPlayed;
			print OUTFILE "\n";
		
			$overalltrack += 1;
		}
	}
	close OUTFILE;
	print STDOUT "\nData written into $outfilename.\n";
	print STDOUT "\n===================================\n";
}
