# 				lastfm2itunes.pl  v0.62(MAY-2012)
#
#    Copyright (c) 2012 Lionel Grenier (lastfm2itunes@gmail.com / lastfm.igrenier.com)
#
#    code for processing lastFM data has been inspired by Jugdizh (lastFM user)
#    code for updating iTunes data has been inspired by  the  iTunesUpdate script for slimserver by James Craig (james.craig@london.com)
#    Update v0.65 
#              - add more verbose info : the overall playcount after processing each weekly file and displayed all found songs.
#              - I let a sleep of 1 sec between polling files which seemed to cause data losse (!!)

#
#    This perl script allows to update iTunes tracks with the following info from a 
#    lastFM user account:
#        - track's play count (if greater than the iTunes ones)
#        - track's last play date
#
#    WARNING: 
#            I STRONGLY RECOMMEND TO BACKUP YOUR iTUNES LIBRARY 
#            PRIOR TO USING THIS SCRIPT (just in case)
#    
#    It can be used to get back play count info if you lost your iTunes library one day
#    and want to rebuild it (which happend to me)
# 
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# needed perl packages
use Class::Struct;
use XML::DOM;
use strict;
use POSIX qw(strftime);
use Encode;
use Time::Local;

binmode(STDOUT, ":utf8");

# script version
my $scriptVersion='v0.65 (MAY-2014)';

# lastFM URL
my $chartlisturl = 'http://ws.audioscrobbler.com/2.0/user/<username>/weeklychartlist.xml';
my $weeklycharturl = 'http://ws.audioscrobbler.com/2.0/user/<username>/weeklytrackchart.xml?from=<start>&to=<end>';

# lastFM username
my $username;
my $parser = new XML::DOM::Parser;

# lastFM data retrieval
my %trackplays;
my %trackdate;

# stats data
my $overAllCount = 0;
my $trackSkip=0;
my $trackNotFound=0;
my $overalltrack=0;
my $startDate='';

# 1: debug messages to ON
# 0: debug messages to OFF
my $verbose = '';

# iTunes is OK
my $ITUNES_HOOK = 0;

# handle for iTunes app
my $iTunesHandle=();

#iTunes Version
my $iTunesFullVersion;
my $iTunesVersion;

# os supported
my $os='win';

my $file='';
my $outfilename='';
my $time=0;

# lastFM track data structure
struct lastFMTrackInfo => {
	# Artist's name for current song
	songArtist => '$',
	# Track title for current song
	songTrack => '$',
	# Album title for current song.
	songAlbum => '$',
	# last played date
	lastPlayed => '$',
    # first played date
	firstPlayed => '$',
	# play count
	playedCount => '$',
};

#############################################################"
# MAIN
print STDOUT "\nlastFM to iTunes track updater [".$scriptVersion."]\n";
print STDOUT "Copyright (c) 2012 Lionel Grenier (lastfm2itunes\@gmail.com - lastfm.igrenier.com)\n";
print STDOUT "(if you have any question or problem do not hesitate to contact the author)\n";

# retrieve lastFM username
$username = $ARGV[0] || die "\nERROR : Please supply your last.fm username as the first argument.";
my $dd = $ARGV[1] || '.';
if ($dd  eq "t") {
	$dd  = " ".$ARGV[2]."/".$ARGV[3]."/".$ARGV[4];
	print STDOUT "Start parsing for week around ".$dd."\n"; 
	$time = timelocal(0,0,0,$ARGV[2],$ARGV[3]-1,$ARGV[4]);
	print STDOUT "Start parsing for week around ".strftime ("%d-%m-%Y", localtime($time));
	
	$verbose = $ARGV[5] || '.';
	if ($verbose eq "v") {
		print STDOUT "\tverbose on\n";
	} else {
		if ($verbose eq "csv") {
			$file="csv";
			$outfilename = $ARGV[6] || die "\nERROR : Please supply a filename for csv output.";
			$verbose="v";
		}
	}
} else {
	$verbose = $ARGV[1] || '.';
	if ($verbose eq "v") {
		print STDOUT "\tverbose on\n";
	} else {
		if ($verbose eq "csv") {
			$file="csv";
			$outfilename = $ARGV[2] || die "\nERROR : Please supply a filename for csv output.";
			$verbose="v";
		}
	}
}


print STDOUT "\n\n";

if ($file eq "csv") {
	processLastFMTracks();
	writeToCSV();
	displayCSVStats();
} else {
	startiTunes() or die $!;
	processLastFMTracks();
	updateiTunes();
	displayStats();
	disconnectiTunes();
}

print STDOUT "\nThanks for using the script. Bye\n";

exit;
#end of main
#############################################################"

#############################################################"
# lastFM sub functions  

sub processLastFMTracks {
	my $url = $chartlisturl;
	$url =~ s/<username>/$username/g;
	print STDOUT "\nURL: ".$url."\n";
	
	my $chartlistdoc = $parser->parsefile($url);
	die "Could not retrieve page" unless $chartlistdoc;
	sleep 1;

	die "Could not find root node <weeklychartlist>" unless (my $rootlist = $chartlistdoc->getElementsByTagName("weeklychartlist",0))->getLength;
	my $chartlist = $rootlist->item(0)->getElementsByTagName("chart",0);
	
	print STDOUT "\n=============================\n";
	print STDOUT "\tParsing lastFM data\n";
	print STDOUT "\tUsing ".$username." lastFM account.\n";
	print STDOUT "\tFound a total of ".$chartlist->getLength." weeks full of played tracks to parse.\n\n";
	
	print STDOUT "\tLook for start date data\n";
	my $idx = -1;
	my $maxEnd = 0;
	
	for (my $i = 0; $i < $chartlist->getLength; $i++) {
	  my $start = $chartlist->item($i)->getAttribute("from") || die "ERROR: Could not find 'from' attribute in <chart> node.";
	  my $end = $chartlist->item($i)->getAttribute("to") || die "ERROR: Could not find 'to' attribute in <chart> node.";
	  $maxEnd = $end;
	  if ( $time<=$start && $idx==-1) {
		$idx=$i;
		OutMsg("Found Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...\n");
	  }
	  
	  if ( $time>=$start && $time<=$end && $idx==-1) {
		$idx=$i;
		OutMsg("Found Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...\n");
	  }
	}
	return print STDOUT "No data after the ".strftime ("%d-%m-%Y", localtime($time))." max available date ".strftime ("%d-%m-%Y", localtime($maxEnd)) unless $idx>-1;
	
	print STDOUT "\tWait while parsing data\n";
	my $nbremain;
	
	for (my $i = $idx; $i < $chartlist->getLength; $i++) {
	  my $start = $chartlist->item($i)->getAttribute("from") || die "ERROR: Could not find 'from' attribute in <chart> node.";
	  my $end = $chartlist->item($i)->getAttribute("to") || die "ERROR: Could not find 'to' attribute in <chart> node.";
	  
	 
	  OutMsg("Week [",$i,"] ".strftime ("%d-%m-%Y", localtime($start))." to ".strftime ("%d-%m-%Y", localtime($end))."...");
	  
	  $url = $weeklycharturl;
	  $url =~ s/<username>/$username/g;
	  $url =~ s/<start>/$start/g;
	  $url =~ s/<end>/$end/g;
	  
	  print STDOUT "\nWeekly URL: ".$url."\n";
	  my $weeklychartdoc;
	  eval { $weeklychartdoc = $parser->parsefile($url); };
	  if (!$weeklychartdoc) {
		return;
	  }
	  #sleep 1;

	  return print STDOUT "Could not find root tag <weeklytrackchart>" unless ($rootlist = $weeklychartdoc->getElementsByTagName("weeklytrackchart",0))->getLength;
	  my $tracklist = $rootlist->item(0)->getElementsByTagName("track",0);
	  my $nbTracks = $tracklist->getLength;
	  
	  OutMsg("got ".$nbTracks." tracks this week\n");
	  my $totalplays = 0;
	  for (my $j = 0; $j < $tracklist->getLength; $j++) {
	    return print STDOUT "Could not find <artist> tag" unless (my $artistlist = $tracklist->item($j)->getElementsByTagName("artist",0))->getLength;
	    return print STDOUT "Could not find <name> tag" unless (my $songlist = $tracklist->item($j)->getElementsByTagName("name",0))->getLength;
	    return print STDOUT "Could not find <playcount> tag" unless (my $playcountlist = $tracklist->item($j)->getElementsByTagName("playcount",0))->getLength;

	    return print STDOUT "Could not get text from <artist> tag"
	      unless $artistlist->item(0)->hasChildNodes && (my $artisttext = $artistlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;
	    return print STDOUT "Could not get text from <name> tag"
	      unless $songlist->item(0)->hasChildNodes && (my $songtext = $songlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;
	    return print STDOUT "Could not get text from <playcount> tag"
	      unless $playcountlist->item(0)->hasChildNodes && (my $playcounttext = $playcountlist->item(0)->getFirstChild)->getNodeType == TEXT_NODE;

		
		my $artist = encode_utf8(lc($artisttext->getData));
		my $song = encode_utf8(lc($songtext->getData)); 
		
		#my $artist = lc($artisttext->getData);
		#my $song = lc($songtext->getData); 
		
	    my $playcount = $playcounttext->getData;
		# keep the first date in lastFM data
		if ($startDate=='' && $playcount>0) {
			$startDate=strftime ("%d-%m-%Y", localtime($start));
			OutMsg(">>>>FOUND when you started scrobbling !!! on".$startDate."\n");
		}
		
		OutMsg("\t".$song." by ".$artist." played " .$playcount." times that week\n");
		
	    if (exists $trackplays{$artist}) {
            if (exists $trackplays{$artist}{$song}) {
                $trackplays{$artist}{$song} += $playcount;
                $trackdate{$artist}{$song}{'END'} = strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
            } else {
                $trackplays{$artist}{$song} = $playcount;
                $trackdate{$artist}{$song}{'START'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($start));
                $trackdate{$artist}{$song}{'END'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
            }
	    } else {
	      $trackplays{$artist}{$song} = $playcount;
          $trackdate{$artist}{$song}{'START'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($start));
          $trackdate{$artist}{$song}{'END'}= strftime ("%Y-%m-%d %H:%M:%S", localtime($end));
	    }
	    $totalplays += $playcount;
	  }
	  
	  $overAllCount += $totalplays;
	  $nbremain = $chartlist->getLength-($i+1);
	  OutMsg("got ".$totalplays." scrobbles this week. Overall ".$overAllCount." so far..." .$nbremain." weeks remain to process\n");
	}
	print STDOUT "\tParsing done\n";
	print STDOUT "=============================\n";
}

# shows current last FM track data
sub showLastFMtrackInfo {
	my $track = shift;

	OutMsg("======= LastFM track Info ========\n");
	OutMsg("Artist:",$track->songArtist(),"\n");
	OutMsg("Track: ",$track->songTrack(),"\n");
	OutMsg("Play Count: ",$track->playedCount(),"\n");
    OutMsg("Frist Play Date: ",$track->firstPlayed(),"\n");
	OutMsg("Last Play Date: ",$track->lastPlayed(),"\n");
	OutMsg("==================================\n");
}

#############################################################"
# iTunes sub functions  

###### iTunes Connection sub
# Check iTunes avaibility and establish a connection
sub startiTunes {
	if ( !$ITUNES_HOOK ) {
		if ($os eq 'win') {
			require Win32::OLE;
			import Win32::OLE;
			#Win32::OLE->Option(Warn => \&OLEError);
			Win32::OLE->Option(Warn => 0);
			Win32::OLE->Option(CP => Win32::OLE::CP_UTF8());
		} else {
			OutErrMsg("This script is not supported on this plateform\n");
			return 0;
		}
		
		_openiTunes() or return 0;
		$ITUNES_HOOK=1;
	
		return 1;
	}
}
#disconnect iTunes connection
sub disconnectiTunes {
	_closeiTunes(); 
	$ITUNES_HOOK=0;
}

# connect to local iTunes
# 1/ load iTunes if not already loaded
# 2/ connect to its database
sub _openiTunes {
	my $failure;

	unless ($iTunesHandle) {
		print STDOUT ("Attempting to make connection to iTunes...\n");
		if ($os eq 'win') {
			$iTunesHandle = new Win32::OLE( 'iTunes.Application') 
		} else {
			print STDOUT ("This is not supported on plattform\n");
			return 0;
		}
		unless ($iTunesHandle) {
			OutErrMsg( "Failed to launch iTunes!!!\n");
			return 0;
		}
		my $iTunesFullVersion = $iTunesHandle->Version;
		print STDOUT "Connection established to iTunes: $iTunesFullVersion\n";
		($iTunesVersion) = split /\./,$iTunesFullVersion;
	} else {
		$iTunesHandle->Version or $failure = 1;	
		if ($failure) {
			print STDOUT  ("iTunes dead: reopening...\n");
			undef $iTunesHandle;
			return _openiTunes();
		}
	}
	return 1;
}

# close iTunes connections
sub _closeiTunes {
		print STDOUT "Disconnected from iTunes\n";
		undef $iTunesHandle;
}

# update iTunes track with lastFM info
sub updateiTunes {

	print STDOUT "\n=============================\n";
	print STDOUT "\tUpdating iTunes\n";

	my @artists = sort(keys %trackplays);
	for my $artist (@artists) {
	  my @songs = sort(keys %{$trackplays{$artist}});
	  for my $song (@songs) {
		my $lastFMtrack = lastFMTrackInfo->new();
		$lastFMtrack->songArtist("$artist");
		$lastFMtrack->songTrack("$song");
		$lastFMtrack->lastPlayed($trackdate{$artist}{$song}{'END'});
		$lastFMtrack->playedCount($trackplays{$artist}{$song});
		
		showLastFMtrackInfo($lastFMtrack);
		my  $iTunesTrack=getTrackFromiTunes($lastFMtrack);
		if ($iTunesTrack){
			logTrackToiTunes($iTunesTrack,$lastFMtrack);
			showiTunestrackInfo($iTunesTrack);
			$overalltrack += 1;
		}
		else {
			OutErrMsg("Not found in iTunes\n");
			$trackNotFound += 1;
		}
	  }
	}
	
	print STDOUT "\tiTunes Update completed\n";
	print STDOUT "=============================\n";
}

###### iTunes tracks sub
# write data into iTunes track from lastFM track
sub logTrackToiTunes {
	my $iTunesTrack = shift;
	my $lastFMTrack = shift;

	if ($iTunesTrack) {
		if ($lastFMTrack->playedCount 
		and $lastFMTrack->playedCount>$iTunesTrack->PlayedCount) {
			OutMsg("\tUpdate track play count and date\n");
			if ($os eq 'win') {
				$iTunesTrack->{PlayedCount} = $lastFMTrack->playedCount;
				$iTunesTrack->{PlayedDate} = $lastFMTrack->lastPlayed;
			}
		} else {
			OutMsg("\tTrack not updated\n");
			print STDOUT "\t...but skipped (iTunes count[",$iTunesTrack->PlayedCount,"] > last.FM count[",$lastFMTrack->playedCount,"])\n";
			$trackSkip += 1;
		}
	} else {
		OutErrMsg("Track: ", $lastFMTrack->songTrack() ," not found in iTunes\n");
	}
}

# shows current iTunes track data
sub showiTunestrackInfo {
		my $track = shift;
	
	my $playedDate;
	
	if ($os eq 'win') {
		$playedDate = $track->PlayedDate->Date("dd-MMM-yyyy")
			." "
			.$track->PlayedDate->Time("HH:mm:ss");
		$playedDate = 'Never Played' if ($playedDate =~ m/30-Dec-1899 00:00:00/);
		
		OutMsg("======= iTunes matching track info ========\n");
		OutMsg("Artist:",$track->Artist,"\n");
		OutMsg("Track: ",$track->Name,"\n");
		OutMsg("Album: ",$track->Album,"\n");
		OutMsg("Play Count: ",$track->PlayedCount,"\n");
		OutMsg("Play Date: ",$playedDate,"\n");
		OutMsg("Location: ",$track->Location,"\n");
		OutMsg("===========================================\n");
	
	}
}

# retrieve iTunes track from lastFM track
sub getTrackFromiTunes {
	my $track = shift;
	# create searchString and remove duplicate/trailing whitespace as well.
	my $searchString = "";

	my $artist = $track->songArtist();
	$searchString .= $artist unless (!$artist or $artist eq 'NO_ARTIST');
	my $album = $track->songAlbum();
	$searchString .= " $album" unless (!$album or $album eq 'NO_ALBUM');
	my $title = $track->songTrack();
	$searchString .= " $title" unless ($title eq 'NO_TITLE'); 

	print STDOUT "Searching iTunes for \"$searchString\" track\n";

	return 0 unless length($searchString) >= 1;

	if ($os eq 'win') {
		return _searchiTunesWin($searchString);
	} 
	return undef;
}

# retrieve iTunes track from a string
sub _searchiTunesWin {
		my $searchString = shift;
	
	# File track 
	my $IITrackKindFile = 1;
	my $ITPlaylistSearchFieldVisible = 1;
	
	my $mainLibrary = $iTunesHandle->LibraryPlaylist;	
	my $trackCollection = $mainLibrary->Search($searchString, $ITPlaylistSearchFieldVisible);
	if ($trackCollection)
	{
		for (my $j = 1; $j <= $trackCollection->Count ; $j++) {
			#check the type
			if ($trackCollection->Item($j)->Kind == $IITrackKindFile)
			{
				#we have the file 
				print STDOUT "\t...found\n";
				return $trackCollection->Item($j);
			} else {
				OutErrMsg("False match:  $searchString\n");
			}
		}
	}
	return undef;
}

#############################################################"
# utils sub functions

# handle MS OLE error message 
sub OLEError {
	my $message=Win32::OLE->LastError();
	$message .= "\n";
	
	print STDOUT $message;
}

# handle output message
sub OutMsg {
	# Parameter - Message to be displayed
	my $message = join '',@_;

	if ($verbose eq "v")
	{
		print STDOUT $message;      
	}
}

# handle output error message
sub OutErrMsg {
	my $message = join '','Error: ',@_;
	print STDOUT $message; 
}

# display some stats about the script
sub displayStats {
	print STDOUT "\n=============================\n";
	print STDOUT "Statistics of lastFM 2 Itunes run for ".$username." user account\n";
	print STDOUT "- ".$overalltrack." tracks processed\n";
	print STDOUT "- ".$trackSkip." tracks skipped iTunes library\n";
	print STDOUT "- ".$trackNotFound." tracks not found in local iTunes library\n";
	print STDOUT "- you have ".$overAllCount." play count on lastFM since ".$startDate."\n";
	print STDOUT "=============================\n";
}

sub displayCSVStats {
	print STDOUT "\n=============================\n";
	print STDOUT "Statistics of lastFM 2 Itunes run for ".$username." user account\n";
	print STDOUT "- ".$overalltrack." tracks processed\n";
	print STDOUT "- you have ".$overAllCount." play count on lastFM since ".$startDate."\n";
	print STDOUT "=============================\n";
}
sub writeToCSV {
	print STDOUT "\n===================================\n";
	print STDOUT "\nWrite tracks info into ".$outfilename." ($file format).\n";
	
	open OUTFILE, ">$outfilename" or die "Could not open file to write";
	my @artists = sort(keys %trackplays);
	print OUTFILE "Artist\tSong\tPlayCount\tFirstTime\tLastTime\n";
	for my $artist (@artists) {
		my @songs = sort(keys %{$trackplays{$artist}});
		for my $song (@songs) {
			my $lastFMtrack = lastFMTrackInfo->new();
			$lastFMtrack->songArtist($artist);
			$lastFMtrack->songTrack($song);
            $lastFMtrack->firstPlayed($trackdate{$artist}{$song}{'START'});
			$lastFMtrack->lastPlayed($trackdate{$artist}{$song}{'END'});
			$lastFMtrack->playedCount($trackplays{$artist}{$song});
		
			print OUTFILE $lastFMtrack->songArtist;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->songTrack;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->playedCount;
            print OUTFILE "\t";
			print OUTFILE $lastFMtrack->firstPlayed;
			print OUTFILE "\t";
			print OUTFILE $lastFMtrack->lastPlayed;
			print OUTFILE "\n";
			
			$overalltrack += 1;
		}
	}
	close OUTFILE;
	print STDOUT "\nData written into $outfilename.\n";
	print STDOUT "\n===================================\n";
}